<?php

namespace App\Services;

use Illuminate\Contracts\Auth\Authenticatable as User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Transaction; // main Transaction model used across controllers
use App\Utils\TransactionUtil;
use Spatie\Activitylog\Models\Activity;
use App\User as AppUser;

class AiContextService
{
    /**
     * Build a compact, permission-scoped context string for the AI
     * from the current user's accessible data relevant to the query.
     */
    public function buildContextForUser(User $user, string $query, array $opts = []): string
    {
        $q = Str::lower($query);
        $parts = [];

        $businessId = (int) ($user->business_id ?? session('business.id'));
        if (!$businessId) {
            return '';
        }

        $permittedLocations = method_exists($user, 'permitted_locations')
            ? $user->permitted_locations($businessId)
            : 'all';

        // Supplier-user detection (to restrict totals to their own supplier)
        $isSupplierUser = false; $supplierContactId = null;
        $forcedSupplierId = isset($opts['page_supplier_id']) ? (int) $opts['page_supplier_id'] : (int) (session('ai.page_supplier_id') ?? 0);
        $forcedSupplierName = isset($opts['page_supplier_name']) ? (string) $opts['page_supplier_name'] : (string) (session('ai.page_supplier_name') ?? '');
        try {
            if (!empty($user->supplier_contact_id)) {
                $supplierContactId = (int) $user->supplier_contact_id;
                // Treat any user with a bound supplier contact as a Supplier for scoping
                $isSupplierUser = true;
            }
            if (method_exists($user, 'getRoleNames')) {
                $r = collect($user->getRoleNames())->map(function($x){ return strtolower(trim($x)); });
                if ($r->contains('supplier') || $r->contains('suppliers') || $r->contains(function($v){ return is_string($v) && str_contains($v, 'supplier'); })) {
                    $isSupplierUser = true;
                }
            }
            $ut = strtolower((string) ($user->user_type ?? ''));
            if (in_array($ut, ['supplier','supplier_user','user_supplier'], true)) { $isSupplierUser = true; }
            $sess = (array) session('user', []);
            if (empty($supplierContactId) && !empty($sess['supplier_contact_id'])) { $supplierContactId = (int) $sess['supplier_contact_id']; }
            if (!$isSupplierUser && !empty($sess['roles'])) {
                $sessRoles = collect((array)$sess['roles'])->map(function($x){ return is_array($x)?strtolower($x['name']??''):strtolower((string)$x); });
                if ($sessRoles->contains('supplier') || $sessRoles->contains('suppliers') || $sessRoles->contains(function($v){ return is_string($v) && str_contains($v, 'supplier'); })) {
                    $isSupplierUser = true;
                }
            }
        } catch (\Throwable $e) {}


                // Fuzzy resolve supplier contact if supplier user without link
        if ($isSupplierUser && empty($supplierContactId)) {
            try {
                $terms = [];
                if (!empty($user->username)) { $terms[] = strtolower(trim((string) $user->username)); }
                $fn = strtolower(trim((string) ($user->first_name ?? '')));
                $ln = strtolower(trim((string) ($user->last_name ?? '')));
                if ($fn || $ln) {
                    $terms[] = trim($fn . ' ' . $ln);
                    $terms[] = trim($ln . ' ' . $fn);
                }
                $sessUser = (array) session('user', []);
                if (!empty($sessUser['first_name']) || !empty($sessUser['last_name'])) {
                    $terms[] = strtolower(trim(((string) ($sessUser['first_name'] ?? '')) . ' ' . ((string) ($sessUser['last_name'] ?? ''))));
                }
                if (!empty($sessUser['username'])) {
                    $terms[] = strtolower(trim((string) $sessUser['username']));
                }
                foreach ($terms as $t) {
                    if (!$t) continue;
                    $row = DB::table('contacts')
                        ->where('business_id', $businessId)
                        ->where(function($w) use ($t){
                            $w->whereRaw('LOWER(name) = ?', [$t])
                              ->orWhereRaw('LOWER(supplier_business_name) = ?', [$t])
                              ->orWhereRaw('LOWER(name) like ?', ['%'.$t.'%'])
                              ->orWhereRaw('LOWER(supplier_business_name) like ?', ['%'.$t.'%']);
                        })
                        ->select('id')
                        ->orderByDesc('id')
                        ->first();
                    if ($row) { $supplierContactId = (int) ($row->id ?? 0); if ($supplierContactId) break; }
                }
                // Persist back to users table only if this contact has purchases (avoid wrong links)
                if (!empty($supplierContactId)) {
                    try {
                        $hasPurch = DB::table('transactions')
                            ->where('business_id', $businessId)
                            ->whereIn('type', ['purchase','purchase_order'])
                            ->where('contact_id', $supplierContactId)
                            ->exists();
                        if ($hasPurch) {
                            DB::table('users')
                                ->where('id', $user->getAuthIdentifier())
                                ->update(['supplier_contact_id' => $supplierContactId]);
                        }
                    } catch (\Throwable $e) { /* ignore write failure */ }
                }
            } catch (\Throwable $e) {}
        }// Admin detection: allow viewing other users' activity
        $isAdmin = false;
        try {
            if (method_exists($user, 'getRoleNames')) {
                $roles = collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); });
                $isAdmin = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin');
                // Many places use Admin#<business_id>
                $isAdmin = $isAdmin || $roles->contains('admin#'.strtolower((string)($user->business_id ?? '')));
            }
            $isAdmin = $isAdmin || (method_exists($user, 'can') && $user->can('superadmin'));
        } catch (\Throwable $e) {}

        // Persist page supplier hints for the session thread
        if ($forcedSupplierId > 0) { session(['ai.page_supplier_id' => $forcedSupplierId]); }
        if ($forcedSupplierName !== '') { session(['ai.page_supplier_name' => $forcedSupplierName]); }

        // Helpers
        $applyLocation = function ($builder) use ($permittedLocations) {
            if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                $builder->whereIn('location_id', $permittedLocations);
            }
            return $builder;
        };
        $extractInvoice = function(string $text): ?string {
            // allows 2025/19205, INV-12345 or long digits
            if (preg_match('/([A-Za-z]{2,5}-\d{3,}|\d{2,4}\/\d{2,6}|\d{5,})/', $text, $m)) {
                return $m[1];
            }
            return null;
        };

        // 1) Latest sale detail (with created_by)
        if (Str::contains($q, ['latest sale', 'latest sell', 'last sale', 'recent sale', 'last invoice'])) {
            $latest = $applyLocation(
                Transaction::where('business_id', $businessId)
                    ->where('type', 'sell')
                    ->where('status', 'final')
            )
                ->orderByDesc('transaction_date')
                ->orderByDesc('id')
                ->select('id', 'invoice_no', 'final_total', 'contact_id', 'transaction_date', 'location_id', 'created_by')
                ->with([
                    'contact:id,name,supplier_business_name',
                    'sales_person:id,first_name,last_name,username'
                ])
                ->first();

            if ($latest) {
                $customer = optional($latest->contact)->name ?: optional($latest->contact)->supplier_business_name;
                $adder = $latest->sales_person ? trim(($latest->sales_person->first_name.' '.$latest->sales_person->last_name)) ?: $latest->sales_person->username : null;
                $parts[] = 'Latest Sale: invoice=' . ($latest->invoice_no ?: ('#' . $latest->id))
                    . ', customer=' . ($customer ?: 'N/A')
                    . ', total=Php ' . number_format((float)$latest->final_total, 2)
                    . ', date=' . Carbon::parse($latest->transaction_date)->toDateTimeString()
                    . ($adder ? (', added_by=' . $adder) : '');
            }
        }

        // 2) Today sales total
        if (Str::contains($q, ['today', "today's", 'todays']) && Str::contains($q, ['sales', 'sale', 'sell']) && Str::contains($q, ['total', 'amount', 'revenue', 'sales'])) {
            $builder = Transaction::where('business_id', $businessId)
                ->where('type', 'sell')
                ->where('status', 'final')
                ->whereDate('transaction_date', Carbon::today());
            $applyLocation($builder);
            if ($isAdmin) {
                $candidate = null;
                if (preg_match('/@([A-Za-z0-9_.-]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                if ($candidate === null && preg_match('/(?:of|by|for)\\s+(?:user|username)?\\s*([A-Za-z0-9._\\-\\s]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                if ($candidate !== null) {
                    $nm = strtolower(trim($candidate));
                    $nm = preg_replace('/\\b(ms|mr|mrs|ma\'?am|maam|sir|madam|user|username)\\.?\\s+/i','', $nm);
                    $u = AppUser::where('business_id', $businessId)
                        ->where(function($uu) use ($nm){
                            $uu->whereRaw('LOWER(username) like ?', ['%'.$nm.'%'])
                               ->orWhereRaw('LOWER(first_name) like ?', ['%'.$nm.'%'])
                               ->orWhereRaw('LOWER(last_name) like ?', ['%'.$nm.'%'])
                               ->orWhereRaw("LOWER(CONCAT(COALESCE(first_name,''),' ',COALESCE(last_name,''))) like ?", ['%'.$nm.'%']);
                        })
                        ->first(['id']);
                    if ($u) { $builder->where('created_by', $u->id); }
                }
            }
            $total = (float) $builder->sum('final_total');
            $count = (int) $builder->count();
            $parts[] = 'Today Sales: total=Php ' . number_format($total, 2) . ', count=' . $count;
        }

        // 2b) This month / Last month sales totals (relaxed matching)
        if (Str::contains($q, ['sales','sale','sell']) && (Str::contains($q, ['total','amount','revenue','count']) || Str::contains($q, ['today',"today's",'this week','week','this month','last month','previous month','month','this year','year to date','ytd','last year','previous year','year']))) {
            $isLastMonth = Str::contains($q, ['last month','previous month']);
            $isThisMonth = !$isLastMonth && Str::contains($q, ['this month','month']);
            $isThisYear  = Str::contains($q, ['this year','year to date','ytd','current year']);
            $isLastYear  = !$isThisYear && Str::contains($q, ['last year','previous year']);
            if ($isLastYear || $isThisYear) {
                $start = $isLastYear ? Carbon::now()->subYearNoOverflow()->startOfYear() : Carbon::now()->startOfYear();
                $end   = Carbon::now(); // year-to-date for "this year" to match typical dashboards

                // If the phrasing is explicit "this year" without ytd, you may want full year
                if ($isThisYear && Str::contains($q, ['full year','entire year','whole year'])) {
                    $end = Carbon::now()->endOfYear();
                }

                $overall = $isAdmin && (Str::contains($q, ['business','overall','all users','everyone','not only me','not just me']));
                $mine = !$overall && (Str::contains($q, ['my','me','mine']) || !$isAdmin);

                $builder = Transaction::where('business_id', $businessId)
                    ->where('type','sell')->where('status','final')
                    ->whereDate('transaction_date','>=',$start->toDateString())
                    ->whereDate('transaction_date','<=',$end->toDateString());
                $applyLocation($builder);
                if ($mine) {
                    $builder->where('created_by', $user->getAuthIdentifier());
                }
                $sum = (float) $builder->sum('final_total');
                $cnt = (int) $builder->count();
                $scope = $overall ? 'Business' : 'Your';
                $endOfYear = Carbon::now()->endOfYear()->toDateString();
                $label = $isLastYear ? 'Last Year' : (($end->toDateString() === $endOfYear) ? 'This Year' : 'This Year (YTD)');
                $parts[] = $label . ' Sales (' . $scope . '): total=Php ' . number_format($sum,2) . ', count=' . $cnt . ', range=' . $start->toDateString() . ' to ' . $end->toDateString();
            } elseif ($isLastMonth || $isThisMonth) {
                $start = $isLastMonth ? Carbon::now()->subMonthNoOverflow()->startOfMonth() : Carbon::now()->startOfMonth();
                $end   = $isLastMonth ? Carbon::now()->subMonthNoOverflow()->endOfMonth() : Carbon::now()->endOfMonth();

                $overall = $isAdmin && (Str::contains($q, ['business','overall','all users','everyone','not only me','not just me']));
                $mine = !$overall && (Str::contains($q, ['my','me','mine']) || !$isAdmin);

                // Optional: explicit user (admins only). Matches "of <name>" or "@username".
                $forUser = null; $forUserLabel = null;
                if ($isAdmin) {
                    $candidate = null;
                    if (preg_match('/@([A-Za-z0-9_.-]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                    if ($candidate === null && preg_match('/(?:of|by|for)\\s+(?:user|username)?\\s*([A-Za-z0-9._\\-\\s]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                    if ($candidate !== null) {
                        $nm = strtolower(trim($candidate));
                        $nm = preg_replace('/\\b(ms|mr|mrs|ma\'?am|maam|sir|madam|user|username)\\.?\\s+/i','', $nm);
                        $tokens = preg_split('/\\s+/', $nm);
                        if (is_array($tokens) && count($tokens) >= 3) { $nm = trim($tokens[count($tokens)-2] . ' ' . $tokens[count($tokens)-1]); }
                        $forUser = AppUser::where('business_id', $businessId)
                            ->where(function($uu) use ($nm){
                                $uu->whereRaw('LOWER(username) like ?', ['%'.$nm.'%'])
                                   ->orWhereRaw('LOWER(first_name) like ?', ['%'.$nm.'%'])
                                   ->orWhereRaw('LOWER(last_name) like ?', ['%'.$nm.'%'])
                                   ->orWhereRaw("LOWER(CONCAT(COALESCE(first_name,''),' ',COALESCE(last_name,''))) like ?", ['%'.$nm.'%']);
                            })
                            ->first(['id','first_name','last_name','username']);
                        if ($forUser) { $forUserLabel = trim(($forUser->first_name.' '.$forUser->last_name)) ?: $forUser->username; }
                    }
                }

                $builder = Transaction::where('business_id', $businessId)
                    ->where('type','sell')->where('status','final')
                    ->whereDate('transaction_date','>=',$start->toDateString())
                    ->whereDate('transaction_date','<=',$end->toDateString());
                $applyLocation($builder);
                if ($forUser) {
                    $builder->where('created_by', $forUser->id);
                } elseif ($mine) {
                    $builder->where('created_by', $user->getAuthIdentifier());
                }
                $sum = (float) $builder->sum('final_total');
                $cnt = (int) $builder->count();
                $scope = $forUser ? ('User '.$forUserLabel . ($forUser && !empty($forUser->username) ? ' (@'.$forUser->username.')' : '')) : ($overall ? 'Business' : 'Your');
                $label = $isLastMonth ? 'Last Month' : 'This Month';
                $parts[] = $label . ' Sales (' . $scope . '): total=Php ' . number_format($sum,2) . ', count=' . $cnt . ', range=' . $start->toDateString() . ' to ' . $end->toDateString();
            }
        }

            // 2c) Purchases totals (business, user, or supplier) for today/this week/this month/last month/year
            if (
                (Str::contains($q, ['purchase','purchases','procurement','bought','buy','po','purchase order'])
               && (Str::contains($q, ['total','amount','spend','spent','cost']) || Str::contains($q, ['today', "today's", 'this week','week','this month','last month','previous month','month','this year','year to date','ytd','last year','previous year','year'])))
            ) {
            // Infer time window with explicit label
            $label = 'This Month';
            $rangeStart = Carbon::now()->startOfMonth();
            $rangeEnd   = Carbon::now()->endOfMonth();
            if (Str::contains($q, ['today', "today's"])) { $label='Today'; $rangeStart = Carbon::today(); $rangeEnd = Carbon::today(); }
            elseif (Str::contains($q, ['this week','week'])) { $label='This Week'; $rangeStart = Carbon::now()->startOfWeek(); $rangeEnd = Carbon::now()->endOfWeek(); }
            elseif (Str::contains($q, ['this year','year to date','ytd','current year'])) {
                $label='This Year (YTD)'; $rangeStart = Carbon::now()->startOfYear(); $rangeEnd = Carbon::now();
                if (Str::contains($q, ['full year','entire year','whole year'])) { $label='This Year'; $rangeEnd = Carbon::now()->endOfYear(); }
            }
            elseif (Str::contains($q, ['last year','previous year'])) { $label='Last Year'; $rangeStart = Carbon::now()->subYearNoOverflow()->startOfYear(); $rangeEnd = Carbon::now()->subYearNoOverflow()->endOfYear(); }
            elseif (Str::contains($q, ['last month','previous month'])) { $label='Last Month'; $rangeStart = Carbon::now()->subMonthNoOverflow()->startOfMonth(); $rangeEnd = Carbon::now()->subMonthNoOverflow()->endOfMonth(); }

            // Supplier resolution priority:
            // 1) If page provided supplier and user is NOT admin => force it
            // 2) Supplier user => their linked supplier
            // 3) Admin may specify supplier by name; otherwise default to page supplier if provided
            $supplier = null; $supplierLabel = null;
            if (!$isAdmin && $forcedSupplierId > 0) {
                $supplier = DB::table('contacts')
                    ->where('business_id', $businessId)
                    ->where('id', $forcedSupplierId)
                    ->select('id','name','supplier_business_name')
                    ->first();
                if ($supplier) { $supplierLabel = ($supplier->name ?: $supplier->supplier_business_name) ?: 'Supplier'; }
            }
            if (!$supplier && !$isAdmin && isset($isSupplierUser) && $isSupplierUser && !empty($supplierContactId)) {
                $supplier = DB::table('contacts')
                    ->where('business_id', $businessId)
                    ->where('id', $supplierContactId)
                    ->select('id','name','supplier_business_name')
                    ->first();
                if ($supplier) { $supplierLabel = ($supplier->name ?: $supplier->supplier_business_name) ?: 'Supplier'; }
            }
            // Admins can optionally specify a supplier name in the question (from/by/for/of)
            if (!$supplier && $isAdmin && preg_match('/(?:from|by|for|of|supplier)\s+([A-Za-z0-9 .\-]{2,80})/i', $q, $m)) {
                $term = strtolower(trim($m[1]));
                $supplier = DB::table('contacts')
                    ->where('business_id', $businessId)
                    ->where(function($w) use ($term){
                        $w->whereRaw('LOWER(name) like ?', ['%'.$term.'%'])
                          ->orWhereRaw('LOWER(supplier_business_name) like ?', ['%'.$term.'%']);
                    })
                    ->select('id','name','supplier_business_name')
                    ->first();
                if ($supplier) {
                    $supplierLabel = ($supplier->name ?: $supplier->supplier_business_name) ?: 'Supplier';
                }
            } elseif (!$supplier && $isAdmin && $forcedSupplierId > 0) {
                $supplier = DB::table('contacts')
                    ->where('business_id', $businessId)
                    ->where('id', $forcedSupplierId)
                    ->select('id','name','supplier_business_name')
                    ->first();
                if ($supplier) { $supplierLabel = ($supplier->name ?: $supplier->supplier_business_name) ?: 'Supplier'; }
            }

            // Build base query over transactions
            $qb = DB::table('transactions as t')
                ->where('t.business_id', $businessId)
                ->whereIn('t.type', ['purchase','purchase_order'])
                ->whereDate('t.transaction_date', '>=', $rangeStart->toDateString())
                ->whereDate('t.transaction_date', '<=', $rangeEnd->toDateString());
            if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                $qb->whereIn('t.location_id', $permittedLocations);
            }
            if ($supplier) { $qb->where('t.contact_id', $supplier->id); }

              // Prefer received purchases; if none, sum whatever exists
              $sumQ = (clone $qb)->where(function($w){ $w->where('t.type', 'purchase')->whereIn('t.status', ['received','completed']); })
                  ->sum('t.final_total');
              $cntQ = (clone $qb)->where(function($w){ $w->where('t.type', 'purchase')->whereIn('t.status', ['received','completed']); })
                  ->count();
              if ($cntQ === 0) {
                  $sumQ = (float) (clone $qb)->sum('t.final_total');
                  $cntQ = (int) (clone $qb)->count();
              }

              // If supplier is set but still zero results, attempt a name-similarity fallback to handle duplicate contacts
              if ($supplier && $cntQ === 0) {
                  try {
                      $nm = strtolower(trim($supplierLabel ?? ''));
                      if ($nm !== '') {
                          $cand = DB::table('contacts as c')
                              ->where('c.business_id', $businessId)
                              ->where(function($w) use ($nm){
                                  $w->whereRaw('LOWER(c.name) like ?', ['%'.$nm.'%'])
                                    ->orWhereRaw('LOWER(c.supplier_business_name) like ?', ['%'.$nm.'%']);
                              })
                              ->select('c.id')
                              ->orderByDesc(DB::raw("(select COALESCE(sum(final_total),0) from transactions t where t.business_id = ".$businessId." and t.contact_id = c.id and t.type in ('purchase','purchase_order'))"))
                              ->first();
                          if ($cand && (int)$cand->id !== (int)$supplier->id) {
                              $supplier = (object)['id' => (int)$cand->id];
                              // Persist corrected link for supplier users
                              if ($isSupplierUser) {
                                  try {
                                      DB::table('users')->where('id', $user->getAuthIdentifier())
                                          ->update(['supplier_contact_id' => $supplier->id]);
                                  } catch (\Throwable $e) {}
                              }
                              $qb2 = DB::table('transactions as t')
                                  ->where('t.business_id', $businessId)
                                  ->whereIn('t.type', ['purchase','purchase_order'])
                                  ->whereDate('t.transaction_date', '>=', $rangeStart->toDateString())
                                  ->whereDate('t.transaction_date', '<=', $rangeEnd->toDateString())
                                  ->where('t.contact_id', $supplier->id);
                              if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                                  $qb2->whereIn('t.location_id', $permittedLocations);
                              }
                              $sumQ = (clone $qb2)->where(function($w){ $w->where('t.type', 'purchase')->whereIn('t.status', ['received','completed']); })
                                  ->sum('t.final_total');
                              $cntQ = (clone $qb2)->where(function($w){ $w->where('t.type', 'purchase')->whereIn('t.status', ['received','completed']); })
                                  ->count();
                              if ($cntQ === 0) { $sumQ = (float) (clone $qb2)->sum('t.final_total'); $cntQ = (int) (clone $qb2)->count(); }
                          }
                      }
                  } catch (\Throwable $e) {}
              }

            // label set explicitly above

            $scope = 'Business';
            if ($supplier) { $scope = 'Supplier ' . $supplierLabel; }
            elseif (Str::contains($q, ['my','me','mine']) && $isAdmin === false) { $scope = 'Your'; }

            $parts[] = $label . ' Purchases (' . $scope . '): total=Php ' . number_format((float)$sumQ,2) . ', count=' . (int)$cntQ . ', range=' . $rangeStart->toDateString() . ' to ' . $rangeEnd->toDateString();
        }

        // 4a) Product prices (group prices). Triggers on price-related phrasing in EN/TL/CEB
        if (
            Str::contains($q, ['price','how much','presyo','magkano','group price','price group','srp','selling price'])
        ) {
            try {
                // find candidate product (reuse logic similar to stocks)
                $prodCandidate = null; $bySku = false;
                if (preg_match('/sku\s*[:#]?\s*([A-Za-z0-9._\-]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; $bySku = true; }
                if ($prodCandidate === null && preg_match('/(?:price|presyo|magkano)\s+(?:for|ng|of)?\s*([A-Za-z0-9 &_\-\(\)\'\"\/\.,]{2,120})/i', $q, $m)) { $prodCandidate = trim($m[1]); }
                if ($prodCandidate === null && preg_match('/how\s*much\s*(?:is\s*(?:the\s*)?)?(?:price\s*)?(?:for|of|ng)?\s*([A-Za-z0-9 &_\-\(\)\'\"\/\.,]{2,120})/i', $q, $m)) { $prodCandidate = trim($m[1]); }
                if ($prodCandidate === null && preg_match('/(?:for|para\s+sa)\s*([A-Za-z0-9 &_\-\(\)\'\"\/\.,]{2,120})\s*(?:price|presyo)?/i', $q, $m)) { $prodCandidate = trim($m[1]); }

                // resolve a target price group label if referenced
                $groups = \DB::table('selling_price_groups')->where('business_id', $businessId)->where('is_active',1)->pluck('name','id');
                $groupId = null; $groupLabel = null;
                foreach ($groups as $gid => $gname) {
                    if (Str::contains($q, strtolower($gname))) { $groupId = (int)$gid; $groupLabel = (string)$gname; break; }
                }
                // common aliases
                if ($groupId === null) {
                    $aliases = [
                        'regular' => 'regular',
                        'end user' => 'end-user', 'end-user' => 'end-user',
                        'credit' => 'credit', 'credit card' => 'credit card',
                        'distributor' => 'distributor'
                    ];
                    foreach ($aliases as $needle => $canonical) {
                        if (Str::contains($q, $needle)) {
                            foreach ($groups as $gid => $gname) { if (Str::contains(strtolower($gname), $canonical)) { $groupId=(int)$gid; $groupLabel=$gname; break 2; } }
                        }
                    }
                }

                if ($prodCandidate) {
                    $term = trim($prodCandidate);
                    // Remove any mentioned price group label or common aliases from the term
                    $stripWords = ['regular','end user','end-user','credit','credit card','distributor','srp','selling price','price','presyo','magkano'];
                    if (!empty($groupLabel)) { $stripWords[] = strtolower($groupLabel); }
                    $tLower = strtolower($term);
                    foreach ($stripWords as $sw) { $tLower = str_ireplace($sw, '', $tLower); }
                    $tLower = preg_replace('/\s+/', ' ', $tLower);
                    $tLower = trim($tLower, " \t\n\r\0\x0B-_/.,()[]{}?");
                    if ($tLower !== '') { $term = $tLower; }
                    $baseQ = \DB::table('products as p')
                        ->join('variations as v', 'v.product_id', '=', 'p.id')
                        ->leftJoin('product_variations as pv', 'pv.id', '=', 'v.product_variation_id')
                        ->where('p.business_id', $businessId);
                    if ($bySku) { $baseQ->where('v.sub_sku', 'like', '%'.$term.'%'); }
                    else {
                        $baseQ->where(function($w) use ($term){
                            $w->where('p.name','like','%'.$term.'%')->orWhere('v.sub_sku','like','%'.$term.'%');
                        });
                    }
                    $vars = $baseQ->limit(5)
                        ->get(['p.id as product_id','p.name as product_name','v.id as variation_id','v.sub_sku','v.sell_price_inc_tax','pv.name as group_name','v.name as variation_name']);
                    if ($vars->count()) {
                        $varIds = $vars->pluck('variation_id')->all();
                        $vgp = \DB::table('variation_group_prices as vgp')
                            ->join('selling_price_groups as spg','spg.id','=','vgp.price_group_id')
                            ->whereIn('vgp.variation_id', $varIds)
                            ->select('vgp.variation_id','spg.name as group_name','vgp.price_inc_tax','vgp.price_type','vgp.price_group_id')
                            ->get();
                        $byVar = [];
                        foreach ($vgp as $r) { $byVar[(int)$r->variation_id][] = $r; }
                        $lines = [];
                        foreach ($vars as $vrow) {
                            $base = (float) $vrow->sell_price_inc_tax;
                            $varName = is_string($vrow->variation_name) ? trim($vrow->variation_name) : '';
                            if ($varName !== '' && preg_match('/^(dummy|default|n\/a|na|none|-|single)$/i', $varName)) { $varName = ''; }
                            $label = trim($vrow->product_name . ($varName ? (' - '.$varName) : '') . ' ['.($vrow->sub_sku ?: '-') . ']');
                            $entries = $byVar[(int)$vrow->variation_id] ?? [];
                            if (empty($entries)) { $lines[] = $label . ' base=Php ' . number_format($base,2); continue; }
                            if ($groupId !== null) {
                                // choose that group only
                                $match = collect($entries)->first(function($x) use ($groupId){ return (int)$x->price_group_id === (int)$groupId; });
                                if ($match) {
                                    $price = ($match->price_type === 'percentage') ? (new \App\Utils\Util())->calc_percentage($base, (float)$match->price_inc_tax) : (float)$match->price_inc_tax;
                                    $lines[] = $label . ' ('.$groupLabel.')=Php ' . number_format($price,2);
                                }
                            } else {
                                // list available groups
                                $pairs = [];
                                foreach ($entries as $e) {
                                    $price = ($e->price_type === 'percentage') ? (new \App\Utils\Util())->calc_percentage($base, (float)$e->price_inc_tax) : (float)$e->price_inc_tax;
                                    $pairs[] = $e->group_name . '=Php ' . number_format($price,2);
                                }
                                if ($pairs) { $lines[] = $label . ' prices: ' . implode(', ', $pairs); }
                            }
                        }
                        if ($lines) { $parts[] = 'Product Prices: ' . implode('; ', $lines); }
                    }
                }
            } catch (\Throwable $e) {}
        }

        // 4b) Product stocks and suppliers (handles English + Filipino/Tagalog phrasing)
        // Triggers on common English words OR Filipino cues like "meron bang", "may stock ba ng", "available ba ang"
        if (
            Str::contains($q, ['product','stock','inventory','qty','quantity','available','availability','in stock','on hand','have'])
            || preg_match('/\b(meron|mayroon)\s+(ba|bang)?\b/i', $q)
            || preg_match('/\bmay\s+(stock\s+)?(ba\s+)?(ng|nang)?\b/i', $q)
            || preg_match('/\bavailable\s+ba\s+ang\b/i', $q)
        ) {
            try {
                // Specific product by name or SKU
                $prodCandidate = null; $bySku = false;
                if (preg_match('/sku\s*[:#]?\s*([A-Za-z0-9._\-]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; $bySku = true; }
                if ($prodCandidate === null && preg_match('/product\s+(?:named|name)?\s*([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                if ($prodCandidate === null && preg_match('/stock\s+of\s+([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                if ($prodCandidate === null && preg_match('/have\s+([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                if ($prodCandidate === null && preg_match('/available\s+([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                if ($prodCandidate === null && preg_match('/of\s+([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                // Filipino: "meron bang <term>", "may stock ba ng <term>", "available ba ang <term>",
                // also tolerate pronouns like "tayong/ating/nating/kaming"
                $stopwords = ['tayong','tayo','kami','kaming','ating','nating','ang','ng','sa','yung','yong','mga','ba','bang','po','ho'];
                if ($prodCandidate === null && preg_match('/(?:meron|mayroon)\s+(?:ba|bang)?\s*(?:tayong|tayo|kami|kaming|ating|nating)?\s*([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                if ($prodCandidate === null && preg_match('/may\s*(?:stock\s*)?(?:ba\s*)?(?:ng|nang)\s*(?:tayong|tayo|kami|kaming|ating|nating)?\s*([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                if ($prodCandidate === null && preg_match('/available\s*ba\s*ang\s*(?:tayong|tayo|kami|kaming|ating|nating)?\s*([A-Za-z0-9 _\-\(\)\'\"]{2,100})/i', $q, $m)) { $prodCandidate = $m[1]; }
                // Last fallback: if the query is short (1-4 words) assume it's a product search term
                if ($prodCandidate === null) {
                    $tokens = preg_split('/\s+/', trim(preg_replace('/[\?\.!]+$/','',$q)));
                    if (is_array($tokens) && count($tokens) > 0 && count($tokens) <= 7) {
                        // Remove leading Filipino stopwords/pronouns
                        $filtered = array_values(array_filter(array_map(function($t){ return strtolower(trim($t)); }, $tokens), function($t) use ($stopwords){ return $t !== '' && !in_array($t, $stopwords, true); }));
                        if (count($filtered) >= 1) {
                            // Use last 1–3 tokens as candidate term
                            $take = min(3, count($filtered));
                            $slice = array_slice($filtered, -$take);
                            $prodCandidate = trim(implode(' ', $slice));
                        }
                    }
                }

                if ($prodCandidate) {
                    $term = trim($prodCandidate);
                    // Find product -> variations -> qty by permitted locations
                    $query = DB::table('products as p')
                        ->join('variations as v', 'v.product_id', '=', 'p.id')
                        ->leftJoin('variation_location_details as vld', 'vld.variation_id', '=', 'v.id')
                        ->where('p.business_id', $businessId);
                      if ($bySku) {
                          $query->where('v.sub_sku', 'like', '%'.$term.'%');
                      } else {
                          $query->where(function($w) use ($term){
                              $w->where('p.name', 'like', '%'.$term.'%')
                                ->orWhere('v.sub_sku', 'like', '%'.$term.'%');
                          });
                      }
                    if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                        $query->whereIn('vld.location_id', $permittedLocations);
                    }
                    $rows = $query->groupBy('p.id','p.name','v.sub_sku','p.is_inactive')
                        ->select('p.id','p.name','v.sub_sku','p.is_inactive', DB::raw('COALESCE(SUM(vld.qty_available),0) as qty'))
                        ->orderByDesc('qty')->limit(5)->get();
                    if ($rows->count()) {
                        $lines = [];
                          foreach ($rows as $r) {
                              // Supplier info intentionally omitted per request.
                              $inactive = (isset($r->is_inactive) && (int)$r->is_inactive === 1) ? ' (inactive)' : '';
                              // Per-location breakdown
                              $locQ = DB::table('variations as v')
                                  ->join('variation_location_details as vld','vld.variation_id','=','v.id')
                                  ->join('business_locations as bl','bl.id','=','vld.location_id')
                                  ->where('v.product_id', $r->id);
                              if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                                  $locQ->whereIn('vld.location_id', $permittedLocations);
                              }
                              $locs = $locQ->groupBy('bl.id','bl.name')
                                  ->select('bl.name', DB::raw('COALESCE(SUM(vld.qty_available),0) as qty'))
                                  ->orderByDesc('qty')->get();
                              $locStr = '';
                              if ($locs->count()) {
                                  $locStr = ' locations: ' . $locs->map(function($l){ return ($l->name ?: 'Loc') . '=' . (float)$l->qty; })->implode(', ');
                              }
                              $lines[] = ($r->name) . $inactive . ' ['.($r->sub_sku ?: '-').'] qty=' . (float)$r->qty . $locStr;
                          }
                        $parts[] = 'Product Stock: ' . implode('; ', $lines);
                    }
                } else {
                    // Overview: first 10 products with qty and optional supplier
                    $q2 = DB::table('products as p')
                        ->join('variations as v','v.product_id','=','p.id')
                        ->leftJoin('variation_location_details as vld','vld.variation_id','=','v.id')
                        ->where('p.business_id', $businessId);
                    if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                        $q2->whereIn('vld.location_id', $permittedLocations);
                    }
                    $list = $q2->groupBy('p.id','p.name','p.is_inactive')
                        ->select('p.id','p.name','p.is_inactive', DB::raw('COALESCE(SUM(vld.qty_available),0) as qty'))
                        ->orderByDesc('qty')->limit(10)->get();
                    if ($list->count()) {
                        $summ = [];
                        foreach ($list as $r) {
                            $inactive = (isset($r->is_inactive) && (int)$r->is_inactive === 1) ? ' (inactive)' : '';
                            $summ[] = $r->name . $inactive . ' qty=' . (float)$r->qty;
                        }
                        $parts[] = 'Inventory Overview (top qty): ' . implode('; ', $summ);
                    }
                }
            } catch (\Throwable $e) {}
        }

        // 3) Today gross profit
        if (Str::contains($q, ['profit', 'gross profit', 'net profit']) && Str::contains($q, ['today', "today's", 'todays'])) {
            try {
                $tu = app(TransactionUtil::class);
                $gp = (float) $tu->getGrossProfit($businessId, Carbon::today()->toDateString(), Carbon::today()->toDateString(), null, null, $permittedLocations);
                $parts[] = 'Today Gross Profit: amount=Php ' . number_format($gp, 2);
            } catch (\Throwable $e) {}
        }

        // 3b) This month / Last month / This year / Last year gross profit (with optional specific user for admins)
        if (Str::contains($q, ['profit', 'gross profit', 'net profit']) && Str::contains($q, ['month','last month','previous month','this month','year','this year','last year','ytd'])) {
            try {
                $isLastMonth = Str::contains($q, ['last month','previous month']);
                $isThisMonth = !$isLastMonth && Str::contains($q, ['this month','month']);
                $isThisYear  = Str::contains($q, ['this year','year to date','ytd','current year']);
                $isLastYear  = !$isThisYear && Str::contains($q, ['last year','previous year']);

                $start = null; $end = null; $label = null;
                if ($isLastMonth) { $label = 'Last Month'; $start = Carbon::now()->subMonthNoOverflow()->startOfMonth(); $end = Carbon::now()->subMonthNoOverflow()->endOfMonth(); }
                elseif ($isThisMonth) { $label = 'This Month'; $start = Carbon::now()->startOfMonth(); $end = Carbon::now()->endOfMonth(); }
                elseif ($isLastYear) { $label = 'Last Year'; $start = Carbon::now()->subYearNoOverflow()->startOfYear(); $end = Carbon::now()->subYearNoOverflow()->endOfYear(); }
                elseif ($isThisYear) { $label = 'This Year (YTD)'; $start = Carbon::now()->startOfYear(); $end = Carbon::now(); if (Str::contains($q, ['full year','entire year','whole year'])) { $label = 'This Year'; $end = Carbon::now()->endOfYear(); } }

                if ($start && $end) {
                    $createdBy = null; $userLabel = null;
                    if ($isAdmin) {
                        $candidate = null;
                        if (preg_match('/@([A-Za-z0-9_.-]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                        if ($candidate === null && preg_match('/(?:of|by|for)\s+(?:user|username)?\s*([A-Za-z0-9._\-\s]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                        if ($candidate !== null) {
                            $nm = strtolower(trim($candidate));
                            $nm = preg_replace('/\b(ms|mr|mrs|ma\'?am|maam|sir|madam|user|username)\.?\s+/i','', $nm);
                            $tokens = preg_split('/\s+/', $nm);
                            if (is_array($tokens) && count($tokens) >= 3) { $nm = trim($tokens[count($tokens)-2] . ' ' . $tokens[count($tokens)-1]); }
                            $u = AppUser::where('business_id', $businessId)
                                ->where(function($uu) use ($nm){
                                    $uu->whereRaw('LOWER(username) like ?', ['%'.$nm.'%'])
                                       ->orWhereRaw('LOWER(first_name) like ?', ['%'.$nm.'%'])
                                       ->orWhereRaw('LOWER(last_name) like ?', ['%'.$nm.'%'])
                                       ->orWhereRaw("LOWER(CONCAT(COALESCE(first_name,''),' ',COALESCE(last_name,''))) like ?", ['%'.$nm.'%']);
                                })
                                ->first(['id','first_name','last_name','username']);
                            if ($u) { $createdBy = $u->id; $userLabel = trim(($u->first_name.' '.$u->last_name)) ?: $u->username; }
                        }
                    }

                    $tu = app(TransactionUtil::class);
                    $gp = (float) $tu->getGrossProfit($businessId, $start->toDateString(), $end->toDateString(), null, $createdBy, $permittedLocations);
                    $suffix = $userLabel ? (' for User '.$userLabel) : '';
                    $parts[] = $label . ' Gross Profit' . $suffix . ': amount=Php ' . number_format($gp, 2) . ', range=' . $start->toDateString() . ' to ' . $end->toDateString();
                }
            } catch (\Throwable $e) {}
        }

        // 4) Low stock / inventory health
        if (Str::contains($q, ['low stock','reorder','inventory alert','stock alert','inventory'])) {
            try {
                $invQ = DB::table('variation_location_details as vld')
                    ->join('variations as v', 'vld.variation_id', '=', 'v.id')
                    ->join('products as p', 'v.product_id', '=', 'p.id')
                    ->where('p.business_id', $businessId)
                    ->whereNotNull('p.alert_quantity');
                if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                    $invQ->whereIn('vld.location_id', $permittedLocations);
                }
                $invQ->select('p.name','v.sub_sku','p.alert_quantity', DB::raw('SUM(vld.qty_available) as qty'))
                    ->groupBy('p.id','v.id','p.name','v.sub_sku','p.alert_quantity')
                    ->havingRaw('qty <= p.alert_quantity')
                    ->orderBy('qty','asc')
                    ->limit(10);
                $rows = $invQ->get();
                if ($rows->count()) {
                    $list = $rows->map(function($r){
                        return ($r->name ?: 'Unknown') . ' [' . ($r->sub_sku ?: '-') . '] qty=' . (float)$r->qty . ' alert=' . (float)$r->alert_quantity;
                    })->implode('; ');
                    $parts[] = 'Low Stock: ' . $list;
                }
            } catch (\Throwable $e) {}
        }

        // 5) Sales payment dues (AR)
        if (Str::contains($q, ['due','dues','receivable','invoice due','payment due'])) {
            try {
                $dueRows = DB::table('transactions as t')
                    ->leftJoin('transaction_payments as p', 'p.transaction_id', '=', 't.id')
                    ->where('t.business_id', $businessId)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->select('t.id','t.invoice_no', DB::raw('t.final_total - IFNULL(SUM(p.amount),0) as due'), 't.location_id')
                    ->groupBy('t.id','t.invoice_no','t.final_total','t.location_id')
                    ->havingRaw('due > 0');
                if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                    $dueRows->whereIn('t.location_id', $permittedLocations);
                }
                $all = $dueRows->get();
                $totalDue = (float) $all->sum('due');
                $countDue = (int) $all->count();
                $sample = $all->sortByDesc('due')->take(5)->map(function($r){
                    return (($r->invoice_no ?: ('#'.$r->id)) . '=Php ' . number_format((float)$r->due, 2));
                })->implode('; ');
                if ($countDue > 0) {
                    $parts[] = 'Receivables: invoices_due=' . $countDue . ', total_due=Php ' . number_format($totalDue, 2) . ', sample: ' . $sample;
                }
            } catch (\Throwable $e) {}
        }

        // 6) Top products (time-bound)
        if (Str::contains($q, ['top products','top product','best sellers','bestsellers','top selling','most sold'])) {
            $start = null; $end = null;
            if (Str::contains($q, ['today', "today's", 'todays'])) {
                $start = Carbon::today()->toDateString(); $end = $start;
            } elseif (Str::contains($q, ['week','this week'])) {
                $start = Carbon::now()->startOfWeek()->toDateString();
                $end = Carbon::now()->endOfWeek()->toDateString();
            } elseif (Str::contains($q, ['month','this month'])) {
                $start = Carbon::now()->startOfMonth()->toDateString();
                $end = Carbon::now()->endOfMonth()->toDateString();
            } else {
                $start = Carbon::now()->subDays(7)->toDateString();
                $end = Carbon::today()->toDateString();
            }
            try {
                $qtp = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->join('products as p', 'tsl.product_id', '=', 'p.id')
                    ->where('t.business_id', $businessId)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '>=', $start)
                    ->whereDate('t.transaction_date', '<=', $end)
                    ->select('p.name', DB::raw('SUM(tsl.quantity - tsl.quantity_returned) as qty'), DB::raw('SUM((tsl.unit_price_inc_tax) * (tsl.quantity - tsl.quantity_returned)) as amount'))
                    ->groupBy('p.id','p.name')
                    ->orderByDesc(DB::raw('qty'))
                    ->limit(5);
                if ($permittedLocations !== 'all' && is_array($permittedLocations) && !empty($permittedLocations)) {
                    $qtp->whereIn('t.location_id', $permittedLocations);
                }
                $rows = $qtp->get();
                if ($rows->count()) {
                    $parts[] = 'Top Products ' . $start . ' to ' . $end . ': ' . $rows->map(function($r){ return $r->name . ' qty=' . (float)$r->qty; })->implode('; ');
                }
            } catch (\Throwable $e) {}
        }

        // 7) Purchase orders / latest purchases (with created_by)
        if (Str::contains($q, ['latest purchase','recent purchase','purchase order','po'])) {
            try {
                $latestPurch = $applyLocation(
                    Transaction::where('business_id', $businessId)
                        ->whereIn('type', ['purchase','purchase_order'])
                )
                    ->orderByDesc('transaction_date')
                    ->orderByDesc('id')
                    ->limit(3)
                    ->with(['sales_person:id,first_name,last_name,username'])
                    ->get(['id','type','ref_no','invoice_no','final_total','transaction_date','created_by']);
                if ($latestPurch->count()) {
                    $parts[] = 'Latest Purchases: ' . $latestPurch->map(function($t){
                        $label = $t->ref_no ?: ($t->invoice_no ?: ('#'.$t->id));
                        $adder = $t->sales_person ? (trim($t->sales_person->first_name.' '.$t->sales_person->last_name) ?: $t->sales_person->username) : null;
                        return ($t->type) . ' ' . $label . ' total=Php ' . number_format((float)$t->final_total, 2) . ' date=' . Carbon::parse($t->transaction_date)->toDateString() . ($adder ? (' added_by=' . $adder) : '');
                    })->implode('; ');
                }
            } catch (\Throwable $e) {}
        }

        // 7b) Who added a specific sale/purchase by invoice/ref
        if (Str::contains($q, ['who','added']) && (Str::contains($q, ['sale','sell']) || Str::contains($q, ['purchase','po','p.o']))) {
            $code = $extractInvoice($q);
            if ($code) {
                $type = Str::contains($q, ['purchase','po','p.o']) ? ['purchase','purchase_order'] : ['sell'];
                $builder = Transaction::where('business_id',$businessId)->whereIn('type',$type);
                $builder->where(function($qb) use ($code) {
                    $qb->where('invoice_no',$code)->orWhere('ref_no',$code)->orWhere('id', intval(preg_replace('/\D/','',$code)) ?: -1);
                });
                $t = $applyLocation($builder)
                    ->with(['sales_person:id,first_name,last_name,username'])
                    ->first(['id','invoice_no','ref_no','type','created_by']);
                if ($t) {
                    $adder = $t->sales_person ? (trim($t->sales_person->first_name.' '.$t->sales_person->last_name) ?: $t->sales_person->username) : null;
                    $label = $t->invoice_no ?: ($t->ref_no ?: ('#'.$t->id));
                    $parts[] = 'Owner: ' . ($t->type) . ' ' . $label . ' added_by=' . ($adder ?: 'unknown');
                }
            }
        }

        // 7c) Admin: who added the latest sale/purchase
        if (Str::contains($q, ['who','added','latest'])) {
            if (Str::contains($q, ['sale','sell'])) {
                $t = $applyLocation(
                    Transaction::where('business_id',$businessId)->where('type','sell')->where('status','final')
                )->orderByDesc('transaction_date')->orderByDesc('id')
                 ->with(['sales_person:id,first_name,last_name,username'])
                 ->first(['id','invoice_no','created_by','transaction_date']);
                if ($t) {
                    $adder = $t->sales_person ? (trim($t->sales_person->first_name.' '.$t->sales_person->last_name) ?: $t->sales_person->username) : null;
                    $parts[] = 'Owner: latest sale ' . ($t->invoice_no ?: ('#'.$t->id)) . ' added_by=' . ($adder ?: 'unknown');
                }
            }
            if (Str::contains($q, ['purchase','po','p.o','purchase order'])) {
                $t = $applyLocation(
                    Transaction::where('business_id',$businessId)->whereIn('type',['purchase','purchase_order'])
                )->orderByDesc('transaction_date')->orderByDesc('id')
                 ->with(['sales_person:id,first_name,last_name,username'])
                 ->first(['id','ref_no','invoice_no','created_by','transaction_date']);
                if ($t) {
                    $adder = $t->sales_person ? (trim($t->sales_person->first_name.' '.$t->sales_person->last_name) ?: $t->sales_person->username) : null;
                    $label = $t->ref_no ?: ($t->invoice_no ?: ('#'.$t->id));
                    $parts[] = 'Owner: latest purchase ' . $label . ' added_by=' . ($adder ?: 'unknown');
                }
            }
        }

        // 8) Users activity (Admin can see others by name; non-admin sees own)
        if (Str::contains($q, ['activity','activities','logs','recent changes','user activity','users activity'])) {
            try {
                $aq = Activity::with('causer:id,first_name,last_name,username,business_id')
                    ->orderByDesc('created_at')
                    ->limit(10);
                // If admin and a specific user is referenced, resolve that user robustly
                $targetUser = null;
                if ($isAdmin) {
                    $candidate = null;
                    if (preg_match('/@([A-Za-z0-9_.-]{2,40})/i', $q, $m)) {
                        $candidate = $m[1];
                    }
                    if ($candidate === null && preg_match('/(?:of|by|for)\s+(?:user|username)?\s*([A-Za-z0-9._\-\s]{2,40})/i', $q, $m)) {
                        $candidate = $m[1];
                    }
                    if ($candidate !== null) {
                        $name = trim($candidate);
                        $ln = strtolower($name);
                        $ln = preg_replace('/\b(ms|mr|mrs|ma\'?am|maam|sir|madam|user|username)\.?\s+/i','', $ln);
                        $name = trim($ln);
                        $tokens = preg_split('/\s+/', $name);
                        if (is_array($tokens) && count($tokens) >= 3) {
                            $name = trim($tokens[count($tokens)-2] . ' ' . $tokens[count($tokens)-1]);
                        }
                        $nameSql = strtolower($name);
                        $targetUser = AppUser::where('business_id', $businessId)
                            ->where(function($uu) use ($nameSql){
                                $uu->whereRaw('LOWER(username) like ?', ['%'.$nameSql.'%'])
                                   ->orWhereRaw('LOWER(first_name) like ?', ['%'.$nameSql.'%'])
                                   ->orWhereRaw('LOWER(last_name) like ?', ['%'.$nameSql.'%'])
                                   ->orWhereRaw("LOWER(CONCAT(COALESCE(first_name,''),' ',COALESCE(last_name,''))) like ?", ['%'.$nameSql.'%']);
                            })
                            ->first(['id','first_name','last_name','username']);
                    }
                }
                $startTime = null; $endTime = null; $rangeLabel = null;
                if (Str::contains($q, ['today', "today's", 'todays'])) {
                    $startTime = Carbon::today(); $endTime = Carbon::tomorrow(); $rangeLabel = 'today';
                } elseif (Str::contains($q, ['this week','week'])) {
                    $startTime = Carbon::now()->startOfWeek(); $endTime = Carbon::now()->endOfWeek()->addDay(); $rangeLabel = 'this week';
                } elseif (Str::contains($q, ['this month','month'])) {
                    $startTime = Carbon::now()->startOfMonth(); $endTime = Carbon::now()->endOfMonth()->addDay(); $rangeLabel = 'this month';
                }

                if ($isAdmin) {
                    $aq->whereHas('causer', function ($q2) use ($businessId, $targetUser) {
                        $q2->where('business_id', $businessId);
                        if ($targetUser) { $q2->where('id', $targetUser->id); }
                    });
                } else {
                    $aq->where('causer_id', $user->getAuthIdentifier())
                       ->where('causer_type', AppUser::class);
                }

                if ($startTime && $endTime) {
                    $aq->whereBetween('created_at', [$startTime, $endTime]);
                }

                $acts = $aq->get();
                if ($acts->count()) {
                    $parts[] = 'Recent Activity: ' . $acts->map(function($a){
                        $u = $a->causer; $name = $u ? (trim(($u->first_name.' '.$u->last_name)) ?: $u->username) : 'user';
                        return ($a->description ?: 'event') . ' by ' . $name . ' at ' . $a->created_at->toDateTimeString();
                    })->implode('; ');
                } elseif ($isAdmin && $targetUser) {
                    // Fallback: summarize transactions by created_by in timeframe
                    $tx = Transaction::where('business_id', $businessId);
                    if ($startTime && $endTime) {
                        $tx->whereBetween('transaction_date', [$startTime->toDateString(), $endTime->toDateString()]);
                    }
                    $tx = $applyLocation($tx);
                    $tx->where('created_by', $targetUser->id);
                    $salesCount = (clone $tx)->where('type','sell')->count();
                    $salesTotal = (float) (clone $tx)->where('type','sell')->sum('final_total');
                    $purchCount = (clone $tx)->whereIn('type',['purchase','purchase_order'])->count();
                    $purchTotal = (float) (clone $tx)->whereIn('type',['purchase','purchase_order'])->sum('final_total');
                    $label = ($targetUser->first_name || $targetUser->last_name) ? trim(($targetUser->first_name.' '.$targetUser->last_name)) : $targetUser->username;
                    $rl = $rangeLabel ? $rangeLabel : 'recently';
                    $parts[] = 'Activity Summary for ' . $label . ' (' . $rl . '): sales=' . $salesCount . ' total=Php ' . number_format($salesTotal,2) . '; purchases=' . $purchCount . ' total=Php ' . number_format($purchTotal,2);
                }
            } catch (\Throwable $e) {}
        }

        // 9) Week totals and profit
        if (Str::contains($q, ['this week','week'])) {
            try {
                $start = Carbon::now()->startOfWeek()->toDateString();
                $end   = Carbon::now()->endOfWeek()->toDateString();
                $builder = Transaction::where('business_id', $businessId)
                    ->where('type','sell')->where('status','final')
                    ->whereDate('transaction_date','>=',$start)
                    ->whereDate('transaction_date','<=',$end);
                $applyLocation($builder);
                // Optional: this week for a specific user
                if ($isAdmin) {
                    $candidate = null;
                    if (preg_match('/@([A-Za-z0-9_.-]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                    if ($candidate === null && preg_match('/(?:of|by|for)\\s+(?:user|username)?\\s*([A-Za-z0-9._\\-\\s]{2,40})/i', $q, $m)) { $candidate = $m[1]; }
                    if ($candidate !== null) {
                        $nm = strtolower(trim($candidate));
                        $nm = preg_replace('/\\b(ms|mr|mrs|ma\'?am|maam|sir|madam|user|username)\\.?\\s+/i','', $nm);
                        $tokens = preg_split('/\\s+/', $nm);
                        if (is_array($tokens) && count($tokens) >= 3) { $nm = trim($tokens[count($tokens)-2] . ' ' . $tokens[count($tokens)-1]); }
                        $u = AppUser::where('business_id', $businessId)
                            ->where(function($uu) use ($nm){
                                $uu->whereRaw('LOWER(username) like ?', ['%'.$nm.'%'])
                                   ->orWhereRaw('LOWER(first_name) like ?', ['%'.$nm.'%'])
                                   ->orWhereRaw('LOWER(last_name) like ?', ['%'.$nm.'%'])
                                   ->orWhereRaw("LOWER(CONCAT(COALESCE(first_name,''),' ',COALESCE(last_name,''))) like ?", ['%'.$nm.'%']);
                            })
                            ->first(['id']);
                        if ($u) { $builder->where('created_by', $u->id); }
                    }
                }
                $sum = (float) $builder->sum('final_total');
                $cnt = (int) $builder->count();
                $parts[] = 'This Week Sales: total=Php ' . number_format($sum,2) . ', count=' . $cnt . ', range=' . $start . ' to ' . $end;
                try {
                    $tu = app(TransactionUtil::class);
                    $gp = (float) $tu->getGrossProfit($businessId, $start, $end, null, null, $permittedLocations);
                    $parts[] = 'This Week Gross Profit: amount=Php ' . number_format($gp,2);
                } catch (\Throwable $e) {}
            } catch (\Throwable $e) {}
        }

        // 10) Fallback small snapshot to help generic Qs
        if (empty($parts)) {
            $latest5 = $applyLocation(
                Transaction::where('business_id', $businessId)
                    ->where('type', 'sell')
                    ->where('status', 'final')
            )
                ->orderByDesc('transaction_date')
                ->limit(5)
                ->get(['id','invoice_no','final_total','transaction_date']);

            if ($latest5->count()) {
                $lines = $latest5->map(function ($t) {
                    return ($t->invoice_no ?: ('#'.$t->id)) . '|' . Carbon::parse($t->transaction_date)->toDateString() . '|' . number_format((float)$t->final_total, 2);
                })->implode('; ');
                $parts[] = 'Recent Sales (invoice|date|total): ' . $lines;
            }
        }

        return trim(implode("\n", array_filter($parts)));
    }
}




